/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright (c) 2015 Adobe Systems Incorporated. All rights reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 */

/*jslint vars: true, node: true, plusplus: true, devel: true, nomen: true, indent: 4, expr: true */
/*global beforeEach: false, afterEach: false, it: false, describe: false, expect: false */

var sinon = require("sinon"),
    Q = require("q"),
    fs = require("fs"),
    UserSettings = require("../UserSettings");


describe("UserSettings", function () {
    "use strict";
    afterEach(function () {
        delete require.cache[require.resolve("../UserSettings/MemoryInterface.js")];
    });
    it("it should default to in memory db if no interface is provided", function() {
        var settingsInterface = {
            set: sinon.spy()
        };

        var settings = new UserSettings();
        settings.set({"hello": "world"});
        expect(settings.get("hello")).to.equal("world");
    });

    it("should use provided setSetting interface", function() {
        var settingsInterface = {
            set: sinon.spy()
        };

        var settings = new UserSettings(settingsInterface);
        settings.set({"hello": "world"});
    
        expect(settingsInterface.set).to.be.called;
        expect(settingsInterface.set).to.be.calledWith({"hello": "world"});
    });

    it("should be able to set using a function with arity 2", function() {
        var settings = new UserSettings();
        settings.set("hello", "world");
        expect(settings.get("hello")).to.eq("world");
    });


    it("should use provided getSetting interface", function() {
        var settingsInterface = require("../UserSettings/MemoryInterface.js");
        
        var settings = new UserSettings(settingsInterface);
        settings.set({"key":  "value"});
        var value = settings.get("key");
    
        expect(value).to.eq("value");
    });

    it("should load from underlying storage the LRU section of settings every time", function() {
        var settingsInterface = require("../UserSettings/MemoryInterface.js");
        settingsInterface.get = sinon.spy();
        var settings = new UserSettings(settingsInterface);
        
        var value = settings.get("key");
        settings.get("key");
        expect(settingsInterface.get).to.be.calledWith("LRU");
        //lru get LRU get
        expect(settingsInterface.get.callCount).to.equal(4);
    });

    it("Should try to load from the LRU over the underlying storage model", function() {
        var settingsInterface = require("../UserSettings/MemoryInterface.js");
        settingsInterface.get = sinon.spy();
        var settings = new UserSettings(settingsInterface);
        settings._LRUCache.get = sinon.stub().returns("hi");
        
        var value = settings.get("key");
        expect(value).to.eq("hi");
        //LRU get
        expect(settingsInterface.get).to.be.calledOnce;
    });

    it("Should try to load from the LRU over the underlying storage model", function() {
        var settingsInterface = require("../UserSettings/MemoryInterface.js");
        settingsInterface.get = sinon.stub().returns("world");
        var settings = new UserSettings(settingsInterface);
        settings._LRUCache.get = sinon.stub().returns(undefined);
        var value = settings.get("hi");
        expect(value).to.eq("world");
        expect(settingsInterface.get).to.be.called;
    });


    it("setCachedValue should set into the LRU and persist the entire LRU object to the underlying storage model", function() {
        var settingsInterface = require("../UserSettings/MemoryInterface.js");
        var settings = new UserSettings(settingsInterface);
        settingsInterface.set = sinon.spy();
        settingsInterface.get = sinon.stub().returns({});
        settings._LRUCache.set = sinon.spy();
        settings._LRUCache.serialize = sinon.stub().returns("hello");
        settings.setCachedValue("hello", "world");
        //should also attempt to load if it hasn't before
        expect(settingsInterface.get).to.be.calledWith("LRU");
        expect(settings._LRUCache.set).to.be.calledWith("hello", "world");
        expect(settingsInterface.set).to.be.calledWith({"LRU": "hello"});
    });
});


describe("in MemoryInterface", function () {
    it("should set and get settings", function() {
        var settingsInterface = require("../UserSettings/MemoryInterface.js");
        settingsInterface.set({"key":  "value"});
        var value = settingsInterface.get("key");
        expect(value).to.eq("value");
        settingsInterface.set({"key":  "value1"});
        value = settingsInterface.get("key");
        expect(value).to.eq("value1");
        settingsInterface.set({"key1":  "value2"});
        value = settingsInterface.get("key1");
        expect(value).to.eq("value2");
    });


    it("should delete a setting and have it persisted", function() {
        var settingsInterface = require("../UserSettings/MemoryInterface.js");
        settingsInterface.set({"key":  "value"});
        settingsInterface.deleteSetting("key");
        var value = settingsInterface.get("key");
        expect(value).to.be.undefined;
    });


    it("should handle nested objects", function() {
        var settingsInterface = require("../UserSettings/MemoryInterface.js");
        var a = {rar: { otherthing: {"hi" : "world"}}};
        settingsInterface.set(a);
        var settings =  settingsInterface.get("rar");
        expect(settings.otherthing.hi).to.equal("world");
    });
});


describe("UserSettingsFileInterface", function () {
    var sandbox = sinon.sandbox.create(),
        fs = require("fs"),
        PathUtils = require("../PathUtils"),
        settingsPath;

    beforeEach(function() {
        settingsPath = "/Users/wyles/extract.json";
        sandbox.stub(PathUtils, "getUserSettingPath").returns(settingsPath);
    });

    afterEach(function () {
        sandbox.restore();
        delete require.cache[require.resolve("../UserSettings/SettingsFileInterface.js")];
    });

    it("should load a file only once on a get call", function() {
        var settingsInterface = require("../UserSettings/SettingsFileInterface.js");
        var readFileSync = sandbox.stub(fs, "readFileSync").returns("{}");
        settingsInterface.get("rar");
        settingsInterface.get("rar1");
        expect(readFileSync.callCount).to.eq(1);
    });

    it("should rethrow an exception that doesnt't have code ENOENT", function() {
        var settingsInterface = require("../UserSettings/SettingsFileInterface.js");
        var readFileSync = sandbox.stub(fs, "readFileSync").throws("TypeError");
        expect(settingsInterface.get.bind(settingsInterface, "rar")).to.throw();
    });

    it("should load a default value and create a file if we don't have a settings file on disk", function() {
        var settingsInterface = require("../UserSettings/SettingsFileInterface.js");
        var readFileSync = sandbox.stub(fs, "readFileSync").throws({code: "ENOENT"});
        var open = sandbox.stub(fs, "openSync").returns(12);
        var close = sandbox.stub(fs, "closeSync");
        settingsInterface.get("rar");
        expect(open).to.be.calledWith(settingsPath,"w+");
        expect(open).to.be.calledBefore(close);
        expect(close).to.be.calledWith(12);
    });

    it("Should return undefined if key not in file", function() {
        var settingsInterface = require("../UserSettings/SettingsFileInterface.js");
        var readFileSync = sandbox.stub(fs, "readFileSync").returns("{}");
        var value = settingsInterface.get("rar");
        expect(value).to.be.undefined;
    });

    it("Should return a key that is in the file", function() {
        var settingsInterface = require("../UserSettings/SettingsFileInterface.js");
        var readFileSync = sandbox.stub(fs, "readFileSync").returns("{\"rar\": \"value\"}");
        var value = settingsInterface.get("rar");
        expect(value).to.eq("value");
    });


    it("should be able to set and get a value and write to the file before calling get", function() {
        var settingsInterface = require("../UserSettings/SettingsFileInterface.js");
        var readFileSync = sandbox.stub(fs, "readFileSync").returns("{}");
        var writeFileSync = sandbox.stub(fs, "writeFileSync");
        settingsInterface.set({"rar": "hello" });
        expect(settingsInterface.get("rar")).to.eq("hello");
        expect(writeFileSync).to.be.calledOnce;
        expect(writeFileSync).to.be.calledWith(settingsPath, "{\"rar\":\"hello\"}");
    });



    it("should be able to set and get a value and write to the file after calling get", function() {
        var settingsInterface = require("../UserSettings/SettingsFileInterface.js");
        var readFileSync = sandbox.stub(fs, "readFileSync").returns("{\"rar\":\"hello\"}");
        var writeFileSync = sandbox.stub(fs, "writeFileSync");
        settingsInterface.get("rar");
        settingsInterface.set({"rar": "hello1" });
        expect(settingsInterface.get("rar")).to.eq("hello1");
        expect(writeFileSync).to.be.calledOnce;
        expect(writeFileSync).to.be.calledWith(settingsPath, "{\"rar\":\"hello1\"}");
    });


    it("should extend an existing setting object", function() {
        var settingsInterface = require("../UserSettings/SettingsFileInterface.js");
        var readFileSync = sandbox.stub(fs, "readFileSync").returns("{\"rar\":\"hello\"}");
        var writeFileSync = sandbox.stub(fs, "writeFileSync");
        settingsInterface.get("rar");
        settingsInterface.set({"rar1": "hello1" });
        expect(settingsInterface.get("rar1")).to.eq("hello1");
        expect(writeFileSync).to.be.calledOnce;
        expect(writeFileSync).to.be.calledWith(settingsPath, '{"rar":"hello","rar1":"hello1"}');
    });


    it("should not blow up if we can not write to the preferences file after an initial load", function() {
        var settingsInterface = require("../UserSettings/SettingsFileInterface.js");
        var readFileSync = sandbox.stub(fs, "readFileSync").returns("{\"rar\":\"hello\"}");
        var writeFileSync = sandbox.stub(fs, "writeFileSync").throws();
        var consoleWarn = sandbox.stub(console, "warn");
        settingsInterface.get("rar");
        settingsInterface.set({"rar1": "hello1" });
        expect(settingsInterface.get("rar1")).to.eq("hello1");
        expect(writeFileSync).to.be.calledOnce;
        expect(writeFileSync).to.be.calledWith(settingsPath, '{"rar":"hello","rar1":"hello1"}');
        expect(consoleWarn).to.be.called;
    });

    it("should delete a setting and have it persisted", function() {
        var settingsInterface = require("../UserSettings/SettingsFileInterface.js");
        var readFileSync = sandbox.stub(fs, "readFileSync").returns("{\"rar\":\"hello\"}");
        var writeFileSync = sandbox.stub(fs, "writeFileSync");
        settingsInterface.deleteSetting("rar");
        expect(writeFileSync).to.be.calledOnce;
        expect(writeFileSync).to.be.calledWith(settingsPath, '{}');
    });

    it("Set should always reload from the underlying storage", function() {
        var setting = "hi";
        var settingsInterface = require("../UserSettings/SettingsFileInterface.js");
        var a = {rar: "hello", otherthing: {"hi" : "world"}};
        var b = {rar: "hi"};
        var readFileSync = sandbox.stub(fs, "readFileSync").returns(JSON.stringify(a));
        var writeFileSync = sandbox.stub(fs, "writeFileSync");
        var settings = settingsInterface.get("otherthing");
        fs.readFileSync.returns(JSON.stringify(b))
        settingsInterface.set("rar", setting);
        settings = settingsInterface.get("rar");
        expect(settings).to.equal(setting);
    });

    it("should not blow up if we can't create the userSettings File and shouldn't attempt to use that file again", function() {
        var settingsInterface = require("../UserSettings/SettingsFileInterface.js");
        var readFileSync = sandbox.stub(fs, "readFileSync").throws({code: "ENOENT"});
        var openSync = sandbox.stub(fs, "openSync").throws();
        var writeFileSync = sandbox.stub(fs, "writeFileSync");
        var consoleWarn = sandbox.stub(console, "warn");
        settingsInterface.get("hello");
        expect(consoleWarn).to.be.called;
        settingsInterface.set("hello", "world");
        expect(fs.writeFileSync).to.not.be.called;
    });

});
