/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright (c) 2015 Adobe Systems Incorporated. All rights reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 */

/*jslint vars: true, node: true, plusplus: true, devel: true, nomen: true, indent: 4, expr:true */
/*global beforeEach: false, afterEach: false, it: false, describe: false, expect: false*/

"use strict";

var sinon = require("sinon"),
    LRUCache = require("../LRUCache");

describe("LRUCache", function () {
    var sandbox = sinon.sandbox.create();

    afterEach(function () {
        sandbox.restore();
    });

    it("defaults to a maximum 1000 entries", function() {
        var cache = new LRUCache();
        expect(cache._maxCacheEntries).to.equal(1000);
    });

    it("sets and gets values", function() {
        var cache = new LRUCache();

        cache.set("key", "value");
        expect(cache.get("key")).to.equal("value");
    });

    it("returns an underlying object suitable for serialize", function() {
        var cache = new LRUCache();

        cache.set("key", "value");
        expect(JSON.stringify(cache.serialize())).to.equal("{\"key\":\"value\"}");
    });

    it("overwrites values", function() {
        var cache = new LRUCache();

        cache.set("key", "value");
        expect(cache.get("key")).to.equal("value");

        cache.set("key", "newValue");
        expect(cache.get("key")).to.equal("newValue");
    });

    it("unsets values", function() {
        var cache = new LRUCache();

        cache.set("key", "value");
        expect(cache.get("key")).to.equal("value");

        cache.unset("key");
        expect(cache.get("key")).to.be.undefined;
    });

    it("ignores unsetting a non-existent key", function() {
        var cache = new LRUCache();

        expect(cache.get("key")).to.be.undefined;
        cache.unset("key");
        expect(cache.get("key")).to.be.undefined;
    });

    it("ignores setting with a null key", function() {
        var cache = new LRUCache();

        cache.set(null, "value");
        expect(cache.get(null)).to.be.undefined;
    });

    it("ignores setting with an undefined key", function() {
        var cache = new LRUCache();

        cache.set(undefined, "value");
        expect(cache.get(undefined)).to.be.undefined;
    });

    it("ignores setting with an undefined value", function() {
        var cache = new LRUCache();

        cache.set("key", undefined);
        expect(cache.get("key")).to.be.undefined;
    });

    it("sets a null value", function() {
        var cache = new LRUCache();

        cache.set("key", null);
        expect(cache.get("key")).to.be.null;
    });

    it("evicts values", function() {
        var cache = new LRUCache(1);

        cache.set("key1", "value1");
        expect(cache.get("key1")).to.equal("value1");

        cache.set("key2", "value2");
        expect(cache.get("key2")).to.equal("value2");
        expect(cache.get("key1")).to.be.undefined;

    });

    it("evicts the oldest values", function() {
        var cache = new LRUCache(3);

        cache.set("key1", "value1");
        cache.set("key2", "value2");
        cache.set("key3", "value3");
        cache.set("key4", "value4");
        cache.set("key5", "value5");

        expect(cache.get("key1")).to.be.undefined;
        expect(cache.get("key2")).to.be.undefined;
        expect(cache.get("key3")).to.equal("value3");
        expect(cache.get("key4")).to.equal("value4");
        expect(cache.get("key5")).to.equal("value5");
    });
    
    
    it("evicts the oldest values and returns the purged key", function() {
        var cache = new LRUCache(3);

        cache.set("key1", "value1");
        cache.set("key2", "value2");
        cache.set("key3", "value3");
        var purgedKey = cache.set("key4", "value4");
        expect(purgedKey).to.eq("key1");
        purgedKey = cache.set("key5", "value5");
        expect(purgedKey).to.eq("key2");
    });

    it("refreshes a key's age when it gets set again", function() {
        var cache = new LRUCache(3);

        cache.set("key1", "value1");
        cache.set("key2", "value2");
        cache.set("key3", "value3");
        cache.set("key1", "value1"); // Set key1 again.
        cache.set("key4", "value4");
        cache.set("key5", "value5");

        expect(cache.get("key2")).to.be.undefined;
        expect(cache.get("key3")).to.be.undefined;
        expect(cache.get("key1")).to.equal("value1");
        expect(cache.get("key4")).to.equal("value4");
        expect(cache.get("key5")).to.equal("value5");
    });

    it("resets", function () {
        var cache = new LRUCache();

        cache.set("key1", "value1");
        cache.set("key2", "value2");

        cache.reset();

        expect(cache.get("key1")).to.be.undefined;
        expect(cache.get("key2")).to.be.undefined;
    });
});
